/*
 * OneShotTimeoutBarrier.java		2008-08-27
 */
package app.util;


/** A {@code OneShotTimeoutBarrier}, like a {@link Token}, is a form
 * of binary semaphore, but with two differences: (a) the
 * {@link #passWhenOpen(long)}/P() operation has a timeout, and (b) it is
 * "one-shot" in that the {@link #passWhenOpen(long)} method should be
 * called by a single thread on a single occasion only.
 */
public final class OneShotTimeoutBarrier {

/** Creates and returns a new one-shot barrier. */
	public static final OneShotTimeoutBarrier newBarrier() {

		return new OneShotTimeoutBarrier();
	}

/** Flag indicating that this barrier has been opened by
 * an invocation of {@link #open()} on it.
 */
	private boolean		isOpen;
/** Flag indicating that this barrier's {@link #passWhenOpen(long)}
 * operation has been timed out.
 */
	private boolean		timedOut;

/** Constructs a new as-yet unreleased barrier.
 */
	private OneShotTimeoutBarrier() {

		this.isOpen = false;
		this.timedOut = false;
	}

/** Acquires the permission to proceed represented by this barrier,
 * blocking if necessary until it is opened by another thread
 * or until the specified timeout has elapsed (more or less), and
 * returns {@code true} if and only if the barrier is open.
 */
	public final synchronized boolean passWhenOpen(long timeout)
	throws InterruptedException {

		if (! this.isOpen) {
			// NB
			// We cannot use this.wait(timeout) here, because
			// that call gives us no way of knowing what caused its
			// completion.
			this.launchTimeoutThread(timeout);
			while (! this.isOpen && ! this.timedOut) {
				this.wait();
			}
		}

		return this.isOpen;
	}

/** Opens this barrier provided it has not already timed out; if a thread
 * is currently blocked on {@link #passWhenOpen(long)}, that thread will
 * be unblocked as a result.
 */
	public final synchronized void open() {

		if (! this.timedOut) {
			this.isOpen = true;
			this.notify();
		}
	}

/** Times out the the (only) {@link #passWhenOpen(long)} call on
 * this barrier, provided the barrier has not been opened since that
 * call started.
 */
	private final synchronized void doTimeOut() {

		if (! this.isOpen) {
			this.timedOut = true;
			this.notify();
		}
		else {
			System.out.println(
				"####  (OneShotTimeoutBarrier: timeout attempted, but too late.)");
		}
	}

/** Creates and launches the time out thread for this barrier's (only)
 * {@link #passWhenOpen()} call, using the given time out period
 * in ms.
 */
	private final void launchTimeoutThread(long timeout) {

		TimeoutThread tothread = new TimeoutThread(timeout);
		tothread.start();
	}

/** Time out thread. */
	private final class TimeoutThread extends Thread {
		private final long	TIMEOUT;
	/** Constructs a new timeout thread with the given timeout duration, in ms. */
		public TimeoutThread(long timeout) {
			this.TIMEOUT = timeout;
		}
	/** Standard {@code Runnable} interface method for the timeout thread. */
		public void run() {
			double t_actual = -1;
			try {
				long t0 = System.nanoTime();
				Thread.sleep(this.TIMEOUT);
				long t1 = System.nanoTime();
				t_actual = (double)(t1 - t0) / (1000 * 1000);
				OneShotTimeoutBarrier.this.doTimeOut();
				//System.out.println("####  Timeout  t="+t_actual+"ms");
			}
			catch (InterruptedException ix) {
				//System.out.println("####  Timeout thread: "+ix);
			}
		}
	}

//	public static void main(String[] args) throws InterruptedException {
//
//		final int TRLS = 7, TTO = 40;//8;
//
//		System.out.println("T-RELEASE="+TRLS+"  T-TIMEOUT="+TTO);
//		OneShotTimeoutBarrier barrier = new OneShotTimeoutBarrier();
//		launchReleaseThread(barrier, TRLS);
//	
//		boolean released = barrier.passWhenOpen(TTO);
//		barrier = null;
//	
//		String description = (released?"normal release":"timed out");
//		System.out.println("Pass barrier: "+description);
//		System.exit(0);
//	}
//
//	private static void launchReleaseThread(
//		final OneShotTimeoutBarrier BARRIER, final int TRLS) {
//		Thread t = new Thread() {
//			public void run() {
//				try { Thread.sleep(TRLS); }
//				catch (InterruptedException ix) { System.out.println(ix); }
//				BARRIER.open();
//			}
//		};
//		t.start();
//	}
}
