/*
 * FrameNumberSpinner.java		2007-05-19
 */
package app.gui;


import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.JFormattedTextField;

import java.text.ParseException;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


import player.JAFramesPlayer;
import player.SignStatusRecord;


/** A {@code JSpinner}, equipped with a cyclic number model and
 * bound to a given {@code JAFramesPlayer}, whose current frame
 * number is synchronized with this spinner.
 */
public class FrameNumberSpinner extends JSpinner {

/** Data model for this spinner. */
	protected final CyclicIntegerModel		FRAMES_MODEL;
/** The player with whose frames this spinner is synchronized. */
	protected JAFramesPlayer				PLAYER = null;

/** Handler for sign status info updates. */
	protected final SignStatusHandler		SIGN_STATUS_HANDLER;
/** Dynamic frame number range manager. */
	protected final FrameNumberRangeManager	FRAMES_RANGE_MANAGER;

/** Sign status data for the associated player. */
	protected SignStatusRecord				signInfo;

/** Flag used to stifle an infinite regress when the owning app makes a
 * change to the frame number spinner's data model (which in turn feeds
 * a model-changed event back to the app).
 */
	protected transient boolean				framesModelChangeIsInternal
											= false;

/** Frame number spinner's change listener -- delegates
 * state-changed event to the spinner's model change method.
 */
	protected final ChangeListener			FRAME_NO_CHANGE_LISTENER =
	new ChangeListener() {
		public void stateChanged(ChangeEvent cevt) {
			FrameNumberSpinner.this.handleFrameModelChange();
		}
	};

/** Action for the ENTER key on this spinner's frame number field. */
	protected final ActionListener			FRAME_NO_ENTER_LISTENER =
	new ActionListener() {
		public void actionPerformed(ActionEvent aevt) {
			FrameNumberSpinner.this.doNumberEnterAction();
		}
	};

/** Constructs a new spinner, using the given sign status info update
 * handler.
 */
	public FrameNumberSpinner(
		SignStatusHandler sshdlr, FrameNumberRangeManager fnrm) {

		super(new FrameNumberSpinner.CyclicIntegerModel( 0, 0, 0, 1));

		this.SIGN_STATUS_HANDLER = sshdlr;
		this.FRAMES_RANGE_MANAGER = fnrm;

		this.FRAMES_MODEL = (CyclicIntegerModel) super.getModel();

		this.FRAMES_MODEL.addChangeListener(FRAME_NO_CHANGE_LISTENER);

		// Get the frame number text field, then set its width and
		// define the action to be taken when ENTER is pressed on it.
		//
		// (It's not entirely clear to me why the latter is necessary,
		// i.e. why the not inconsiderable JFormattedTextField
		// infrastructure does not by default produce a suitable
		// response to the ENTER key. And, given that apparently it is
		// necessary, I don't know whether or not the following is the
		// most appropriate way to achieve the desired effect.)
		JFormattedTextField nedtxtfld =
			((JSpinner.NumberEditor) this.getEditor()).getTextField();
		nedtxtfld.setColumns(4);
		nedtxtfld.addActionListener(FRAME_NO_ENTER_LISTENER);
	}

/** Sets the frames-player for this spinner, should only be called once,
 * at set up time.
 */
	public void setPlayer(JAFramesPlayer player) {

		// This should only be done once; it would be done in the
		// constructor were it not that the player may not exist
		// at that stage.
		if (this.PLAYER == null) {
			this.PLAYER = player;
		}
	}

/** Prepares to deal with a new animation on the associated player,
 * by creating a new sign status record and resetting the model with
 * the new maximum frame number.
 */
	public void startNewAnimation() {

		// Create a sign-status record for the new animation scanner.
		this.signInfo = this.PLAYER.makeSignStatusRecord();

//		final int N_FRAMES = this.PLAYER.countFrames();
//		this.resetModel(N_FRAMES - 1);
	}

/** Resets the model to its neutral position in which the maximum
 * frame number is zero, effectively disabling it.
 */
	public void resetModelToNeutral() {

		this.resetModel(0);
	}

/** Resets the model with the given maximum frame number. */
	public void resetModel(int fmax) {

		// Set the model's maximum value with the "internal" flag set,
		// allowing us to suppress any response to a model-change event.
		boolean oldintflag = this.framesModelChangeIsInternal;
		this.framesModelChangeIsInternal = true;
		//####
		this.FRAMES_MODEL.setValue(0);
		this.FRAMES_MODEL.setMaximum(fmax);
		this.FRAMES_MODEL.clearOneSignRange();
		//####
		this.framesModelChangeIsInternal = oldintflag;
	}

	public void internalSetOneSignRange(int fmin, int fmax) {

		boolean oldintflag = this.framesModelChangeIsInternal;
		this.framesModelChangeIsInternal = true;
		//####
		this.FRAMES_MODEL.setOneSignRange(fmin, fmax);
		//####
		this.framesModelChangeIsInternal = oldintflag;
	}

	public void internalClearOneSignRange() {

		boolean oldintflag = this.framesModelChangeIsInternal;
		this.framesModelChangeIsInternal = true;
		//####
		this.FRAMES_MODEL.clearOneSignRange();
		//####
		this.framesModelChangeIsInternal = oldintflag;
	}

/** Sets the model's value to the given frame number, flagging the
 * change as internal, in order to prevent the change being propagated
 * to the player in {@link #handleFrameModelChange()}.
 */
	public void internalAdjustMaxValue(int fmax) {
		
		boolean oldintflag = this.framesModelChangeIsInternal;
		this.framesModelChangeIsInternal = true;
		//####
		this.FRAMES_MODEL.setMaximum(fmax);
		//####
		this.framesModelChangeIsInternal = oldintflag;
	}

/** Sets the model's value to the given frame number, flagging the
 * change as internal, in order to prevent the change being propagated
 * to the player in {@link #handleFrameModelChange()}.
 */
	public void internalSetValue(int frameno) {

		boolean oldintflag = this.framesModelChangeIsInternal;
		this.framesModelChangeIsInternal = true;
		//####
		this.FRAMES_MODEL.setValue(frameno);
		//####
		this.framesModelChangeIsInternal = oldintflag;
	}

/** Handler method for the "frame number model changed" event.
 * This method ignores an internally generated model change, but
 * responds to a user- generated one by updating the player and the sign
 * status.
 */
	protected void handleFrameModelChange() {

		// We get the player to show a new frame only if the event is
		// user-generated via the GUI, rather than internally generated
		// during auto-play, when the spinner is used as a frame-index
		// display mechanism.
		if (! this.framesModelChangeIsInternal) {

			// Get the frame index.
			int f = (Integer) this.FRAMES_MODEL.getValue();

			// Play this frame, trying to get the associated sign data.
			this.PLAYER.showFrame(f, this.signInfo);

			// See if we have sign data.
			if (this.signInfo != null) {

				// Extract the values from the sign-data record.
				int slimit = this.signInfo.signLimit();
				int s = this.signInfo.sign();
				String gloss = this.signInfo.gloss();

				// Update our status panel with these values.
				this.updateSignInfo(slimit, s, gloss);
			}
		}

		// Update the number range for the new frame.
		if (this.FRAMES_RANGE_MANAGER != null) {
			this.FRAMES_RANGE_MANAGER.updateFrameNumberRange();
		}
	}

/** Handler for ENTER key applied to this spinner's frame number
 * text field: attempts to update the spinner model by committing the
 * edit, but resets the text field's value if this update fails.
 */
	protected void doNumberEnterAction() {

		JFormattedTextField nedtxtfld =
			((JSpinner.NumberEditor)this.getEditor()).getTextField();
		try {
			nedtxtfld.commitEdit();
		}
		catch (ParseException px) {
			// If new value is no good, then revert to the current one.
			Object okval = this.FRAMES_MODEL.getValue();
			nedtxtfld.setValue(okval);
		}
	}

/** Shows sign-related information in the status panel. */
	protected void updateSignInfo(int slimit, int s, String gloss) {

		if (this.SIGN_STATUS_HANDLER != null) {
			this.SIGN_STATUS_HANDLER.updateSignStatus(slimit, s, gloss);
		}
	}

/** Interface for the dynamic management of the range constraint on
 * this spinner's frame number.
 */
	public static interface FrameNumberRangeManager {
	/** Updates the manager's frame number range in response to a change
	 * in this spinner's frame number model.
	 */
		void updateFrameNumberRange();
	}

/** Interface defining the sign status update operation. */
	public static interface SignStatusHandler {
	/** Provides notification of a sign status update, with the given
	 * sign limit index, current index and gloss name.
	 */
		void updateSignStatus(int slimit, int s, String gloss);
	}

/** Cyclic integer value data model for our frame-index spinner control.
 * This implementation allows the value range to be constrained to those
 * valid for a single sign.
 */
	protected static class CyclicIntegerModel extends SpinnerNumberModel {
	/** Minimum value in the model's single-sign range if one is in force,
	 * otherwise negative.
	 */
		private int			oneSignMin;
	/** Maximum value in the model's single-sign range if one is in force,
	 * otherwise negative.
	 */
		private int			oneSignMax;
	/** Constructs a new Cyclic Integer Model. */
		public CyclicIntegerModel(int val, int min, int max, int stpsz) {
			super(val, min, max, stpsz);
			this.clearOneSignRange();
		}
	/** Indicates if a single-sign range is in force for this model. */
		private boolean inOneSignMode()	{ return (0 <= this.oneSignMin); }
	/** Sets the temporary range values to those given. */
		public void setOneSignRange(int tmin, int tmax) {
			this.oneSignMin = tmin;
			this.oneSignMax = tmax;
		}
		public void clearOneSignRange() {
			this.setOneSignRange(-1, -1);
		}
	/** Returns the next value, cycling from Max to Min if need be. */
		public Object getNextValue() {
			int newval = -1;
			if (this.inOneSignMode()) {
				int val = (Integer) super.getNumber();
				newval = (val == this.oneSignMax ? this.oneSignMin : val + 1);
			}
			else {
				Integer next = (Integer) super.getNextValue();
				if (next == null) {
					next = (Integer) super.getMinimum();
				}
				newval = next;
			}
			return newval;
		}
	/** Returns the previous value, cycling from Min to Max if need be. */
		public Object getPreviousValue() {
			int newval = -1;
			if (this.inOneSignMode()) {
				int val = (Integer) super.getNumber();
				newval = (val == this.oneSignMin ? this.oneSignMax : val - 1);
			}
			else {
				Integer prev = (Integer) super.getPreviousValue();
				if (prev == null) {
					prev = (Integer) super.getMaximum();
				}
				newval = prev;
			}
			return newval;
		}
	}
}
