<xsl:transform version="1.0"
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

<!--  Ralph Elliott  <mailto:re@cmp.uea.ac.uk>  2003-08  -->

<!-- THESE OUTPUT SETTINGS MAY BE OVERRIDDEN BY THE G2H PROCESSOR: -->

<xsl:output method="xml" omit-xml-declaration="yes"
    indent="no" encoding="UTF-8"/>


<!--=================================-->
<!--######## Utility Methods ########-->
<!--=================================-->


<xsl:template name="doDirection8">
    <xsl:param name="dir"/>
    <xsl:param name="pfx"/>

    <xsl:variable name="n" select="string-length($dir)"/>
    <xsl:if test="$n=1">
      <xsl:element name="{concat($pfx,$dir)}"/>
    </xsl:if>
    <xsl:if test="$n=2">
      <xsl:element name="{concat($pfx,substring($dir,1,1))}"/>
      <xsl:element name="{concat($pfx,substring($dir,2,1))}"/>
    </xsl:if>
</xsl:template>


<xsl:template name="doDirection26">
    <xsl:param name="dir"/>
    <xsl:param name="pfx"/>

    <xsl:variable name="n" select="string-length($dir)"/>
    <xsl:if test="$n!=3">
      <xsl:element name="{concat($pfx,$dir)}"/>
    </xsl:if>
    <xsl:if test="$n=3">
      <xsl:element name="{concat($pfx,substring($dir,1,2))}"/>
      <xsl:element name="{concat($pfx,substring($dir,2,2))}"/>
    </xsl:if>
</xsl:template>


<xsl:template name="doHandShape">
    <xsl:param name="hs"/>

    <xsl:choose>
      <xsl:when test="$hs='flat'">
        <xsl:element name="hamflathand"/></xsl:when>
      <xsl:otherwise>
        <xsl:element name="{concat('ham',$hs)}"/></xsl:otherwise>
    </xsl:choose>
</xsl:template>

<xsl:template name="doBend">
    <xsl:param name="b"/>

    <xsl:choose>
      <xsl:when test="$b='bent'">
        <xsl:element name="hamfingerstraightmod"/></xsl:when>
      <xsl:when test="$b='round'">
        <xsl:element name="hamfingerbendmod"/></xsl:when>
      <xsl:when test="$b='hooked'">
        <xsl:element name="hamfingerhookedmod"/></xsl:when>
      <xsl:when test="$b='dblbent'">
        <xsl:element name="hamdoublebent"/></xsl:when>
      <xsl:when test="$b='dblhooked'">
        <xsl:element name="hamdoublehooked"/></xsl:when>
      <!-- hope this one never happens: -->
      <xsl:when test="$b='halfbent'">
        <xsl:element name="hamERROR">
          <xsl:attribute name="message">
            <xsl:value-of select="'Bad fingerbend: halfbent'"/>
           </xsl:attribute>
         </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:element name="hamERROR">
          <xsl:attribute name="message">
            <xsl:value-of select="concat('Bad fingerbend: ',$b)"/>
           </xsl:attribute>
         </xsl:element>
        </xsl:otherwise>
    </xsl:choose>
</xsl:template>


<xsl:template name="doThumbPos">
    <xsl:param name="tp"/>

    <xsl:choose>
      <xsl:when test="$tp='out'">
        <xsl:element name="hamthumboutmod"/></xsl:when>
      <xsl:when test="$tp='across'">
        <xsl:element name="hamthumbacrossmod"/></xsl:when>
      <xsl:when test="$tp='opposed' or $tp='open'">
        <xsl:element name="hamthumbopenmod"/></xsl:when>
      <xsl:otherwise>
        <xsl:element name="hamERROR">
          <xsl:attribute name="message">
            <xsl:value-of select="concat('Bad thumbpos ',$tp)"/>
           </xsl:attribute>
         </xsl:element>
       </xsl:otherwise>
    </xsl:choose>
</xsl:template>


<xsl:template name="doCeeOpening">
    <xsl:param name="co"/>

    <xsl:choose>
      <xsl:when test="$co='tight'">
        <xsl:element name="hamthumbacrossmod"/></xsl:when>
      <xsl:when test="$co='slack'">
        <xsl:element name="hamthumbopenmod"/></xsl:when>
      <xsl:otherwise>
        <xsl:element name="hamERROR">
          <xsl:attribute name="message">
            <xsl:value-of select="concat('Bad ceeopening ',$co)"/>
           </xsl:attribute>
         </xsl:element>
       </xsl:otherwise>
    </xsl:choose>
</xsl:template>


<xsl:template name="doFingerDigits">
    <xsl:param name="dd"/>

    <xsl:if test="contains($dd,'2')">
      <xsl:element name="hamindexfinger"/></xsl:if>
    <xsl:if test="contains($dd,'3')">
      <xsl:element name="hammiddlefinger"/></xsl:if>
    <xsl:if test="contains($dd,'4')">
      <xsl:element name="hamringfinger"/></xsl:if>
    <xsl:if test="contains($dd,'5')">
      <xsl:element name="hampinky"/></xsl:if>
</xsl:template>


<xsl:template name="doThumbDigit">
    <xsl:param name="dd"/>

    <xsl:if test="contains($dd,'1')">
      <xsl:element name="hamthumb"/></xsl:if>
</xsl:template>


<xsl:template name="checkOneSpecialFinger">
    <xsl:param name="f"/>

    <xsl:if test="contains(@specialfingers,$f)">
      <xsl:call-template name="doFingerDigits">
        <xsl:with-param name="dd" select="$f"/>
      </xsl:call-template>
      <xsl:if test="@opposedfinger=$f">
        <xsl:element name="hamlargemod"/>
      </xsl:if>
    </xsl:if>
</xsl:template>


<xsl:template name="checkSpecialFingers">

    <xsl:if test="@specialfingers">
      <xsl:call-template name="doSpecialFingers" />
    </xsl:if>
</xsl:template>


<xsl:template name="doSpecialFingers">

	<xsl:call-template name="checkOneSpecialFinger">
	  <xsl:with-param name="f" select="2"/>
	</xsl:call-template>
	<xsl:call-template name="checkOneSpecialFinger">
	  <xsl:with-param name="f" select="3"/>
	</xsl:call-template>
	<xsl:call-template name="checkOneSpecialFinger">
	  <xsl:with-param name="f" select="4"/>
	</xsl:call-template>
	<xsl:call-template name="checkOneSpecialFinger">
	  <xsl:with-param name="f" select="5"/>
	</xsl:call-template>
</xsl:template>


<xsl:template name="doExemptedNotSpecialFingers">
	<xsl:variable name="exnotsp">
		<xsl:call-template name="setDiffFingers">
			<xsl:with-param name="ffa" select="@exempteddigits" />
			<xsl:with-param name="ffb" select="@specialfingers" />
		</xsl:call-template>
	</xsl:variable>
	<xsl:if test="$exnotsp">
      <xsl:call-template name="doFingerDigits">
        <xsl:with-param name="dd" select="$exnotsp" />
      </xsl:call-template>
	</xsl:if>
</xsl:template>


<xsl:template name="setDiffFingers">
	<xsl:param name="ffa" />
	<xsl:param name="ffb" />

	<xsl:if test=
		"contains($ffa,'2') and not(contains($ffb,'2'))">2</xsl:if>
	<xsl:if test=
		"contains($ffa,'3') and not(contains($ffb,'3'))">3</xsl:if>
	<xsl:if test=
		"contains($ffa,'4') and not(contains($ffb,'4'))">4</xsl:if>
	<xsl:if test=
		"contains($ffa,'5') and not(contains($ffb,'5'))">5</xsl:if>
</xsl:template>


<xsl:template name="doDigits">
    <xsl:param name="dd"/>

    <xsl:call-template name="doThumbDigit">
      <xsl:with-param name="dd" select="$dd"/>
    </xsl:call-template>
    <xsl:call-template name="doFingerDigits">
      <xsl:with-param name="dd" select="$dd"/>
    </xsl:call-template>
</xsl:template>


<xsl:template name="checkFingerBending">
    <xsl:param name="d"/>

    <xsl:variable name="bendattr" select="@*[name()=concat('bend',$d)]"/>
    <xsl:if test="$bendattr!=''">
      <xsl:call-template name="doDigits">
        <xsl:with-param name="dd" select="$d"/>
      </xsl:call-template>
      <xsl:call-template name="doBend">
        <xsl:with-param name="b" select="$bendattr"/>
      </xsl:call-template>
    </xsl:if>
</xsl:template>


<xsl:template name="checkAllFingerBendings">

    <xsl:call-template name="checkFingerBending">
      <xsl:with-param name="d" select="1"/></xsl:call-template>
    <xsl:call-template name="checkFingerBending">
      <xsl:with-param name="d" select="2"/></xsl:call-template>
    <xsl:call-template name="checkFingerBending">
      <xsl:with-param name="d" select="3"/></xsl:call-template>
    <xsl:call-template name="checkFingerBending">
      <xsl:with-param name="d" select="4"/></xsl:call-template>
    <xsl:call-template name="checkFingerBending">
      <xsl:with-param name="d" select="5"/></xsl:call-template>
</xsl:template>


<xsl:template name="checkAllFingerCrossings">

    <xsl:if test="@contactpair">
      <xsl:call-template name="doDigits">
        <xsl:with-param name="dd" select="substring(@contactpair,1,1)"/>
      </xsl:call-template>
      <xsl:element name="{concat('hamfinger',@contactkind)}"/>
      <xsl:call-template name="doDigits">
        <xsl:with-param name="dd" select="substring(@contactpair,2,1)"/>
      </xsl:call-template>
      <xsl:if test="@second_contactpair">
        <xsl:call-template name="doDigits">
          <xsl:with-param
              name="dd" select="substring(@second_contactpair,1,1)"/>
        </xsl:call-template>
        <xsl:element name="{concat('hamfinger',@second_contactkind)}"/>
        <xsl:call-template name="doDigits">
          <xsl:with-param
              name="dd" select="substring(@second_contactpair,2,1)"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>
</xsl:template>


<xsl:template name="checkThumbContact">

    <xsl:if test="@thumbcontact">
      <xsl:element name="{concat('hamfinger',@thumbcontact)}"/>
    </xsl:if>
</xsl:template>


<xsl:template name="checkThumbBetween">

    <xsl:if test="@thumbbetween">
      <xsl:call-template name="doDigits">
        <xsl:with-param name="dd" select="substring(@thumbbetween,1,1)"/>
      </xsl:call-template>
      <xsl:element name="hambetween"/>
      <xsl:call-template name="doDigits">
        <xsl:with-param name="dd" select="substring(@thumbbetween,2,1)"/>
      </xsl:call-template>
    </xsl:if>
</xsl:template>


<xsl:template name="doExtFingerDir">
    <xsl:param name="efd"/>

    <xsl:call-template name="doDirection26">
      <xsl:with-param name="dir" select="$efd"/>
      <xsl:with-param name="pfx" select="'hamextfinger'"/>
    </xsl:call-template>
</xsl:template>


<xsl:template name="doPalmOr">
    <xsl:param name="po"/>

    <xsl:element name="{concat('hampalm',$po)}"/>
</xsl:template>


<xsl:template name="doSide">
    <xsl:param name="sd"/>
    <xsl:param name="arm" select="left"/>

    <!--
    NB The caller has already decided whether to put a hamlr... item
    before or after the relevant site.
    -->
    <xsl:choose>
      <xsl:when test="$sd='left_at' or $sd='right_at'">
        <xsl:element name="hamlrat"/></xsl:when>
      <xsl:when test="$sd='left_beside' or $sd='right_beside'">
        <xsl:element name="hamlrbeside"/></xsl:when>
      <xsl:when test="$sd='radial' or $sd='ulnar'">        
        <xsl:element name="hamlrbeside"/></xsl:when>
      <xsl:when test="$sd='dorsal'">
        <xsl:element name="hamhandback"/></xsl:when>
      <xsl:when test="$sd='palmar'">
        <xsl:element name="hampalm"/></xsl:when>
      <xsl:when test="$sd='back'">
        <xsl:element name="hambehind"/></xsl:when>
      <!-- front is the default, so generate nothing for it: -->
      <xsl:when test="$sd='front'"></xsl:when>
      <xsl:otherwise>
        <xsl:element name="hamERROR">
          <xsl:attribute name="message">
            <xsl:value-of select="concat('Bad side ',$sd)"/>
           </xsl:attribute>
         </xsl:element>
       </xsl:otherwise>
    </xsl:choose>
</xsl:template>


<xsl:template name="doSiteBodyArm">
    <xsl:param name="loc"/>

    <!-- 2010-12:
         Bug fix for missing "ham" prefixes in main site
         and upper/lower diacritic.
    -->
    <xsl:choose>
      <xsl:when test=
          "(starts-with($loc,'upper') or starts-with($loc,'lower')) and
          not(contains($loc,'arm'))">
        <!-- Main site: -->
        <xsl:choose>
          <xsl:when test="contains($loc,'lip')">
            <xsl:element name="hamlips"/></xsl:when>
          <xsl:when test="contains($loc,'eyelid')">
            <xsl:element name="hameyes"/></xsl:when>
          <xsl:when test="contains($loc,'teeth')">
            <xsl:element name="hamteeth"/></xsl:when>
        </xsl:choose>
        <!-- Upper/Lower diacritic: -->
        <xsl:if test="starts-with($loc,'upper')">
          <xsl:element name="hamdoublebent"/></xsl:if>
        <xsl:if test="starts-with($loc,'lower')">
          <xsl:element name="hamdoublehooked"/></xsl:if>
      </xsl:when>
    <!--
    2011-01
    Omit this attempt to convert HNS-3 to HNS-4, since the main
    h->g conversion does not yet recognise it.

      <xsl:when test="$loc='elbow'">
        <xsl:element name="hamelbowinside"/>
        <xsl:element name="hambehind"/>
      </xsl:when>
    -->
      <xsl:otherwise>
        <xsl:element name="{concat('ham',$loc)}"/></xsl:otherwise>
    </xsl:choose>
</xsl:template>


<xsl:template name="doSiteHand">
    <xsl:param name="loc"/>

    <xsl:choose>
      <xsl:when test="$loc='pad'">
        <xsl:element name="hamfingernail"/>
        <xsl:element name="hampalm"/>
      </xsl:when>
      <xsl:when test="$loc='elbow'">
        <xsl:element name="hamelbowinside"/>
        <xsl:element name="hambehind"/>
      </xsl:when>
      <xsl:when test=
          "$loc='tip' or $loc='nail' or $loc='midjoint' or $loc='base'">
        <xsl:element name="{concat('hamfinger',$loc)}"/></xsl:when>
      <!-- OK in HNS-3, but invalid in HNS-4 -->
      <xsl:when test="$loc='side'">
        <xsl:element name="hamfingerside"/></xsl:when>
      <xsl:otherwise>
        <xsl:element name="{concat('ham',$loc)}"/></xsl:otherwise>
    </xsl:choose>      
</xsl:template>


<xsl:template name="doSize">
    <xsl:param name="sz"/>

    <xsl:if test="$sz='small'"><xsl:element name="hamsmallmod"/></xsl:if>
    <xsl:if test="$sz='big'"><xsl:element name="hamlargemod"/></xsl:if>
</xsl:template>


<xsl:template name="checkSize">

    <xsl:if test="@size">
      <xsl:call-template name="doSize">
        <xsl:with-param name="sz" select="@size"/>
      </xsl:call-template>
    </xsl:if>
</xsl:template>


<xsl:template name="checkSecondSize">

    <xsl:if test="@second_size">
      <xsl:call-template name="doSize">
        <xsl:with-param name="sz" select="@second_size"/>
      </xsl:call-template>
    </xsl:if>
</xsl:template>


<xsl:template name="checkCurve">

    <xsl:if test="@curve">
      <xsl:call-template name="doDirection8">
        <xsl:with-param name="dir" select="@curve"/>
        <xsl:with-param name="pfx" select="'hamarc'"/>
      </xsl:call-template>
      <xsl:if test="@curve_size">
        <xsl:call-template name="doSize">
          <xsl:with-param name="sz" select="@curve_size"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>
</xsl:template>


<xsl:template name="checkZigZag">

    <xsl:if test="@zigzag_style">
      <xsl:element name="{concat('ham',@zigzag_style)}"/>
      <xsl:if test="@zigzag_size">
        <xsl:call-template name="doSize">
          <xsl:with-param name="sz" select="@zigzag_size"/>
        </xsl:call-template>
      </xsl:if>
      <xsl:if test="@zigzag_incrdecr">
        <xsl:element name="{concat('ham',@zigzag_incrdecr)}"/>
        <xsl:if test="@zigzag_incrdecr_size">
          <xsl:call-template name="doSize">
            <xsl:with-param name="sz" select="@zigzag_incrdecr_size"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:if>
    </xsl:if>
</xsl:template>


<xsl:template name="checkIncrDecr">
    <xsl:if test="@incrdecr">
      <xsl:element name="{concat('ham',@incrdecr)}"/>
      <xsl:if test="@incrdecr_size">
        <xsl:call-template name="doSize">
          <xsl:with-param name="sz" select="@incrdecr_size"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>
</xsl:template>


<xsl:template name="doDirectedMove">
    <xsl:param name="dir"/>

    <xsl:call-template name="doDirection26">
      <xsl:with-param name="dir" select="$dir"/>
      <xsl:with-param name="pfx" select="'hammove'"/>
    </xsl:call-template>
</xsl:template>


<xsl:template name="doCircularMove">
    <xsl:param name="axis"/>

    <xsl:call-template name="doDirection26">
      <xsl:with-param name="dir" select="$axis"/>
      <xsl:with-param name="pfx" select="'hamcircle'"/>
    </xsl:call-template>
</xsl:template>


<xsl:template name="doBasicRepetition">
    <xsl:param name="rep"/>

    <xsl:variable name="hrep">
      <xsl:choose>
        <xsl:when test="$rep='fromstart' or $rep='continue' or $rep='reverse'">
          <xsl:value-of select="$rep"/>
        </xsl:when>
        <xsl:when test="$rep='fromstart_several'">fromstartseveral</xsl:when>
        <xsl:when test="$rep='continue_several'">continueseveral</xsl:when>
        <!-- TBD: deal with to-fro-to case (and in H2S also?) -->
        <xsl:otherwise></xsl:otherwise> <!-- ...(i.e. empty string) -->
      </xsl:choose>
    </xsl:variable>
    <xsl:if test="$hrep!=''">
      <xsl:element name="{concat('hamrepeat',$hrep)}"/>
    </xsl:if>

    <!-- Error case: -->
    <xsl:if test="$hrep=''">
      <xsl:element name="ERROR">
        <xsl:attribute name="message">
          <xsl:value-of select="concat('bad repetition: ',$hrep)"/>
        </xsl:attribute>
      </xsl:element>
    </xsl:if>
</xsl:template>

<xsl:template name="doApproxRepetition">
    <xsl:choose>
      <xsl:when test="@ellipse_direction">
        <xsl:element name="hamseqbegin"/>
        <xsl:element name="hametc"/>
        <xsl:element name="hamseqend"/>
      </xsl:when>
	  <xsl:otherwise>
        <xsl:element name="hametc"/>
      </xsl:otherwise>
	</xsl:choose>
</xsl:template>


<xsl:template name="checkManner">

<!--  2004-03
      The first of these is now superseded by the following five:
-->
    <xsl:if test="@manner">
      <xsl:element name="{concat('ham',@manner)}"/>
    </xsl:if>

    <xsl:if test="@fast='true'">
      <xsl:element name="hamfast"/>
    </xsl:if>

    <xsl:if test="@slow='true'">
      <xsl:element name="hamslow"/>
    </xsl:if>

    <xsl:if test="@tense='true'">
      <xsl:element name="hamtense"/>
    </xsl:if>

    <xsl:if test="@rest='true'">
      <xsl:element name="hamrest"/>
    </xsl:if>

    <xsl:if test="@halt='true'">
      <xsl:element name="hamhalt"/>
    </xsl:if>
</xsl:template>


<!--
2010-12:  Added checks for:
          - Bounce in space, i.e. for hamlargemod after hamhalt.
		  - Bounce on contact, in targetted motion.

2011-04:  - Relax requirement for halt in the first?
          - Relax requirement for touch in the second?
-->
<xsl:template name="checkBounceInSpace">

    <xsl:if test="(@halt='true') and (@bouncing='true')">
      <xsl:element name="hamlargemod"/>
    </xsl:if>
</xsl:template>


<xsl:template name="checkBounceOnContact">

    <xsl:if test="(*/@bouncing='true') and (*/@contact='touch')">
      <xsl:element name="hamlargemod"/>
    </xsl:if>
</xsl:template>


<!--
2011-01
Provide a named template for the alternating motion attributes, making
it easy to check for them more systematically than hitherto.
-->
<xsl:template name="checkAlternatingRepetitions">

	<xsl:if test="@alternating='true'">
      <xsl:element name="hamalternatingmotion"/>
      <xsl:if test="@second_alternating='true'">
        <xsl:element name="hamalternatingmotion"/>
      </xsl:if>
    </xsl:if>
</xsl:template>


<!--==================================================-->
<!--######## MATCH A Hamgestural_sign ELEMENT ########-->
<!--==================================================-->

<!-- Currently, we expect our input to be a single <hamgestural_sign>
-->
<xsl:template match="/">

    <xsl:apply-templates select="*"/>
</xsl:template>


<!-- We use this only if we are doing an entire SiGML document.
<xsl:template match="sigml">

    <xsl:element name="sigml">
      <xsl:apply-templates select="hamgestural_sign"/>
    </xsl:element>
</xsl:template>
-->


<!--######## hamgestural_sign ########-->
<!--==================================-->
<xsl:template match="hamgestural_sign">

    <xsl:element name="hns_sign">
        <xsl:if test="@gloss">
            <xsl:attribute name="gloss">
                <xsl:value-of select="@gloss"/>
            </xsl:attribute>
        </xsl:if>
        <xsl:element name="hamnosys_manual">
          <!-- Convert sign body to HNS manual element sequence. -->
          <xsl:apply-templates select="*"/>
        </xsl:element>
    </xsl:element>
</xsl:template>


<!--######## sign_nonmanual ########-->
<!--================================-->
<xsl:template match="sign_nonmanual">

    <!-- Just ignore non-manuals for now. -->
</xsl:template>


<!--######## sign_manual ########-->
<!--=============================-->
<xsl:template match="sign_manual">

    <xsl:choose>
      <xsl:when test="sign_manual">
        <!-- Special case - sign is a sequence of (manual) signs -->
        <xsl:for-each select="sign_manual">
          <xsl:if test="position()!=1">
            <xsl:element name="hamplus"/>
          </xsl:if>
          <xsl:apply-templates select="."/>
        </xsl:for-each>
      </xsl:when>
      <xsl:otherwise>
        <!-- Normal case - a single manual sign: -->
        <xsl:if test="@holdover='true'">
          <xsl:element name="hamnomotion"/>
        </xsl:if>
        <xsl:if test="@nondominant='true'">
          <xsl:element name="hamnondominant"/>
        </xsl:if>
        <xsl:if test="@both_hands='true'">
          <xsl:choose>
            <xsl:when test="@lr_symm='true'">
              <xsl:element name="hamsymmlr"/></xsl:when>
            <xsl:otherwise>
              <xsl:element name="hamsymmpar"/></xsl:otherwise></xsl:choose>
          <xsl:if test="@ud_symm='true'">
            <xsl:element name="hamlargemod"/></xsl:if>
          <xsl:if test="@oi_symm='true'">
            <xsl:element name="hamfingerstraightmod"/></xsl:if>
          <xsl:if test="@outofphase='true'">
            <xsl:element name="hamalternatingmotion"/></xsl:if>
        </xsl:if>
        <xsl:apply-templates select="*"/>
      </xsl:otherwise>
    </xsl:choose>
</xsl:template>


<!--######## split_handconfig ########-->
<!--==================================-->
<xsl:template match="split_handconfig">

    <!--
    We process <split_handconfig>s in contiguous groups, each identified
    by its first element.  Hence, we only do anything here in the case
    where the current element is the first in such a group group.
    -->
    <xsl:if test="name(preceding-sibling::*[1])!='split_handconfig'">
      <xsl:element name="hamparbegin"/>
      <!-- Dominant handconfig attributes for this <split_handconfig> group. -->
      <xsl:apply-templates select="self::*" mode="dom"/>
      <!-- Fix up empty <handconfig>s in split replacements: -->
      <xsl:if test="parent::tgt_motion and not(handconfig[1]/@*)">
        <xsl:element name="hamnomotion"/>
      </xsl:if>
      <xsl:element name="hamplus"/>
      <!-- Non-dom. handconfig attributes for this <split_handconfig> group. -->
      <xsl:apply-templates select="self::*" mode="nondom"/>
      <!-- Fix up empty <handconfig>s in split replacements: -->
      <xsl:if test="parent::tgt_motion and not(handconfig[2]/@*)">
        <xsl:element name="hamnomotion"/>
      </xsl:if>
      <xsl:element name="hamparend"/>
    </xsl:if>
</xsl:template>


<xsl:template match="split_handconfig" mode="dom">

    <!--
    First, extract info from our dominant <handconfig>, and then
    recursively do the same for any immediately following <split_handconfig>s.
    -->
    <xsl:apply-templates select="handconfig[1]"/>
    <xsl:apply-templates
          select="following-sibling::*[1][name()='split_handconfig']"
          mode="dom"/>
</xsl:template>


<xsl:template match="split_handconfig" mode="nondom">

    <!--
    First, extract info from our non-dominant <handconfig>, and then
    recursively do the same for any immediately following <split_handconfig>s.
    -->
    <xsl:apply-templates select="handconfig[2]"/>
    <xsl:apply-templates
          select="following-sibling::*[1][name()='split_handconfig']"
          mode="nondom"/>
</xsl:template>


<!--######## handconfig (non-manual) ########-->
<!--============================-->
<xsl:template match="handconfig" mode="nonman">

    <!-- Just do the main site for now. Side TBD. -->
    <xsl:call-template name="doSiteBodyArm">
      <xsl:with-param name="loc" select="@bodypart"/>
    </xsl:call-template>
</xsl:template>


<!--######## handconfig ########-->
<!--============================-->
<xsl:template match="handconfig">

    <!-- Basic Handshape: -->
    <xsl:if test="@handshape">
      <xsl:call-template name="doHandShape">
        <xsl:with-param name="hs" select="@handshape"/>
      </xsl:call-template>
    </xsl:if>

    <!-- Added in 2011-04: -->
    <xsl:if test="@approx_shape='true'">
      <xsl:element name="hametc"/>
    </xsl:if>

	<xsl:if test="@mainbend">
      <xsl:call-template name="doBend">
        <xsl:with-param name="b" select="@mainbend"/>
      </xsl:call-template>
    </xsl:if>

    <xsl:if test="@thumbpos">
      <xsl:call-template name="doThumbPos">
        <xsl:with-param name="tp" select="@thumbpos"/>
      </xsl:call-template>
    </xsl:if>

    <xsl:if test="@ceeopening">
      <xsl:call-template name="doCeeOpening">
        <xsl:with-param name="co" select="@ceeopening"/>
      </xsl:call-template>
    </xsl:if>

    <xsl:if test="@second_handshape">

      <xsl:element name="hambetween"/>

      <xsl:call-template name="doHandShape">
        <xsl:with-param name="hs" select="@second_handshape"/>
      </xsl:call-template>

      <xsl:if test="@second_mainbend">
        <xsl:call-template name="doBend">
          <xsl:with-param name="b" select="@second_mainbend"/>
        </xsl:call-template>
      </xsl:if>

      <xsl:if test="@second_thumbpos">
        <xsl:call-template name="doThumbPos">
          <xsl:with-param name="tp" select="@second_thumbpos"/>
        </xsl:call-template>
      </xsl:if>

      <xsl:if test="@second_ceeopening">
        <xsl:call-template name="doCeeOpening">
          <xsl:with-param name="co" select="@second_ceeopening"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>

	<!--  2010-12 ...  -->
    <!-- Finger list: -->
    <!--xsl:call-template name="checkSpecialFingers"/-->
    <xsl:if test="@specialfingers">
        <xsl:call-template name="doSpecialFingers" />
    </xsl:if>
    <xsl:if test="@exempteddigits">
      <!-- Just do fingers here (thumb is dealt with later): -->
	  <!--
	        if not(@specialfingers): just do @exempteddigits
	        if (@specialfingers):  do any exempted not in special
	  -->
	  <xsl:if test="not(@specialfingers)">
        <xsl:call-template name="doFingerDigits">
          <xsl:with-param name="dd" select="@exempteddigits" />
        </xsl:call-template>
      </xsl:if>
	  <xsl:if test="@specialfingers">
        <xsl:call-template name="doExemptedNotSpecialFingers" />
      </xsl:if>
    </xsl:if>

    <xsl:call-template name="checkAllFingerBendings"/>

    <xsl:call-template name="checkAllFingerCrossings"/>

    <!-- Thumb specials: -->
    <xsl:if test="contains(@exempteddigits,'1')">
      <xsl:call-template name="doThumbDigit">
        <xsl:with-param name="dd" select="'1'"/>
       </xsl:call-template>
    </xsl:if>
    <xsl:if test="@thumbenclosed">
      <xsl:element name="hambetween"/>
    </xsl:if>
    <xsl:call-template name="checkThumbBetween"/>
    <xsl:call-template name="checkThumbContact"/>
      
    <!-- Extended Finger Direction: -->
    <xsl:if test="@extfidir">
      <xsl:call-template name="doExtFingerDir">
        <xsl:with-param name="efd" select="@extfidir"/>
      </xsl:call-template>
      <xsl:if test="@second_extfidir">
        <xsl:element name="hambetween"/>
        <xsl:call-template name="doExtFingerDir">
          <xsl:with-param name="efd" select="@second_extfidir"/>
        </xsl:call-template>
      </xsl:if>
      <xsl:if test="@rel_extfidir='true'">
        <xsl:element name="hamorirelative"/>
      </xsl:if>
    </xsl:if>

    <!-- Palm Orientation: -->
    <xsl:if test="@palmor">
      <xsl:call-template name="doPalmOr">
        <xsl:with-param name="po" select="@palmor"/>
      </xsl:call-template>
      <!-- Added in 2011-04: -->
      <xsl:if test="@approx_palmor='true'">
        <xsl:element name="hametc"/>
      </xsl:if>
      <xsl:if test="@second_palmor">
        <xsl:element name="hambetween"/>
        <xsl:call-template name="doPalmOr">
          <xsl:with-param name="po" select="@second_palmor"/>
        </xsl:call-template>
      </xsl:if>
      <xsl:if test="@rel_palmor='true'">
        <xsl:element name="hamorirelative"/>
      </xsl:if>
    </xsl:if>
</xsl:template>


<!--######## split_location ########-->
<!--================================-->
<xsl:template match="split_location">

    <xsl:element name="hamparbegin"/>
    <xsl:apply-templates select="*[1]"/>
    <xsl:element name="hamplus"/>
    <xsl:apply-templates select="*[2]">
      <xsl:with-param name="arm" select="right"/>
    </xsl:apply-templates>
    <xsl:element name="hamparend"/>
</xsl:template>


<!--######## handconstellation ########-->
<!--===================================-->
<xsl:template match="handconstellation">

    <!-- Pair of hand-locations if required: -->
    <xsl:if test="location_hand">
      <xsl:element name="hamparbegin"/>
        <xsl:apply-templates select="*[1]"/>
      <xsl:element name="hamplus"/>
      <xsl:apply-templates select="*[2]">
        <xsl:with-param name="arm" select="right"/>
      </xsl:apply-templates>
      <xsl:element name="hamparend"/>
    </xsl:if>
    <!-- Contact (mandatory for a handconstellation): -->
    <xsl:element name="{concat('ham',@contact)}"/>
    <!-- Body-location if required: -->
    <xsl:apply-templates select="location_bodyarm"/>
</xsl:template>


<!--######## location_bodyarm ########-->
<!--==================================-->
<xsl:template match="location_bodyarm">
    <xsl:param name="arm" select="left"/>

    <!-- Do side as prefix if one is needed: -->
    <xsl:if test=
        "starts-with(@side,'left') or
        (@side='radial' and $arm='right') or
        (@side='ulnar' and $arm='left')">
      <xsl:call-template name="doSide">
        <xsl:with-param name="sd" select="@side"/>
        <xsl:with-param name="arm" select="$arm"/>
      </xsl:call-template>
    </xsl:if>
    <!-- Do the main site: -->
    <xsl:call-template name="doSiteBodyArm">
      <xsl:with-param name="loc" select="@location"/>
    </xsl:call-template>
    <!-- Added in 2011-04: -->
    <xsl:if test="@approx_location='true'">
      <xsl:element name="hametc"/>
    </xsl:if>
    <!-- Do side as suffix if one is needed: -->
    <xsl:if test=
        "@side and
        not(starts-with(@side,'left')) and
        not(@side='radial' and $arm='right') and
        not(@side='ulnar' and $arm='left')">
      <xsl:call-template name="doSide">
        <xsl:with-param name="sd" select="@side"/>
        <xsl:with-param name="arm" select="$arm"/>
      </xsl:call-template>
    </xsl:if>
    <!-- Test for seccond location (for betweenness): -->
    <xsl:if test="@second_location">
      <xsl:element name="hambetween"/>
      <!-- Do side as prefix if one is needed: -->
      <xsl:if test=
          "starts-with(@second_side,'left') or
          (@second_side='radial' and $arm='right') or
          (@second_side='ulnar' and $arm='left')">
        <xsl:call-template name="doSide">
          <xsl:with-param name="sd" select="@second_side"/>
          <xsl:with-param name="arm" select="$arm"/>
        </xsl:call-template>
      </xsl:if>
      <!-- Do the main site: -->
      <xsl:call-template name="doSiteBodyArm">
        <xsl:with-param name="loc" select="@second_location"/>
      </xsl:call-template>
      <!-- Do side as suffix if one is needed: -->
      <xsl:if test=
          "@second_side and
          not(starts-with(@second_side,'left')) and
          not(@second_side='radial' and $arm='right') and
          not(@second_side='ulnar' and $arm='left')">
        <xsl:call-template name="doSide">
          <xsl:with-param name="sd" select="@second_side"/>
          <xsl:with-param name="arm" select="$arm"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>
    <!-- Possible "behind" flag: -->
    <xsl:if test="@behind='true'">
      <xsl:element name="hambehind"/>
    </xsl:if>
    <!-- Do contact: -->
    <xsl:if test="@contact">
      <xsl:if test="not(*)">
        <xsl:element name="{concat('ham',@contact)}"/>
      </xsl:if>
      <xsl:if test="location_hand">
        <xsl:element name="hamseqbegin"/>
        <xsl:element name="{concat('ham',@contact)}"/>
        <xsl:apply-templates select="location_hand">
          <!-- ???? Is this bit right?: -->
          <xsl:with-param name="arm">
            <xsl:if test="$arm='left'">right</xsl:if>
            <xsl:if test="$arm='right'">left</xsl:if>
          </xsl:with-param>
        </xsl:apply-templates>
        <xsl:element name="hamseqend"/>
      </xsl:if>
    </xsl:if>
</xsl:template>


<!--######## location_hand ########-->
<!--===============================-->
<xsl:template match="location_hand">
    <xsl:param name="arm" select="left"/>

    <!-- Primary location attributes: -->
    <!-- Do side as prefix if one is needed: -->
    <xsl:if test=
        "starts-with(@side,'left') or
        (@side='radial' and $arm='right') or
        (@side='ulnar' and $arm='left')">
      <xsl:call-template name="doSide">
        <xsl:with-param name="sd" select="@side"/>
        <xsl:with-param name="arm" select="$arm"/>
      </xsl:call-template>
    </xsl:if>
    <!-- Do any fingers/thumb followed by the main site if there is one: -->
    <xsl:if test="@digits">
      <xsl:call-template name="doDigits">
        <xsl:with-param name="dd" select="@digits"/>
      </xsl:call-template>
    </xsl:if>
    <xsl:if test="@location">
      <xsl:call-template name="doSiteHand">
        <xsl:with-param name="loc" select="@location"/>
      </xsl:call-template>
    </xsl:if>
    <!-- Do side as suffix if one is needed: -->
    <xsl:if test=
        "@side and
        not(starts-with(@side,'left')) and
        not(@side='radial' and $arm='right') and
        not(@side='ulnar' and $arm='left')">
      <xsl:call-template name="doSide">
        <xsl:with-param name="sd" select="@side"/>
        <xsl:with-param name="arm" select="$arm"/>
      </xsl:call-template>
    </xsl:if>

    <!-- "second_" location attributes (indicating betweenness): -->
    <xsl:if test="@second_digits or @second_location">
      <xsl:element name="hambetween"/>
      <!-- Do second_side as prefix if one is needed: -->
      <xsl:if test=
          "starts-with(@second_side,'left') or
          (@second_side='radial' and $arm='right') or
          (@second_side='ulnar' and $arm='left')">
        <xsl:call-template name="doSide">
          <xsl:with-param name="sd" select="@second_side"/>
          <xsl:with-param name="arm" select="$arm"/>
        </xsl:call-template>
      </xsl:if>
      <!-- Do any fingers/thumb followed by the site if there is one: -->
      <xsl:if test="@second_digits">
        <xsl:call-template name="doDigits">
          <xsl:with-param name="dd" select="@second_digits"/>
        </xsl:call-template>
      </xsl:if>
      <xsl:if test="@second_location">
        <xsl:call-template name="doSiteHand">
          <xsl:with-param name="loc" select="@second_location"/>
        </xsl:call-template>
      </xsl:if>
      <!-- Do second_side as suffix if one is needed: -->
      <xsl:if test=
          "@second_side and
          not(starts-with(@second_side,'left')) and
          not(@second_side='radial' and $arm='right') and
          not(@second_side='ulnar' and $arm='left')">
        <xsl:call-template name="doSide">
          <xsl:with-param name="sd" select="@second_side"/>
          <xsl:with-param name="arm" select="$arm"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>

    <!-- Do contact: -->
    <xsl:if test="@contact">
      <xsl:if test="not(*)">
        <xsl:element name="{concat('ham',@contact)}"/>
      </xsl:if>
      <xsl:if test="location_hand">
        <xsl:element name="hamseqbegin"/>
        <xsl:element name="{concat('ham',@contact)}"/>
        <xsl:apply-templates select="location_hand">
          <xsl:with-param name="arm">
            <xsl:if test="$arm='left'">right</xsl:if>
            <xsl:if test="$arm='right'">left</xsl:if>
          </xsl:with-param>
        </xsl:apply-templates>
        <xsl:element name="hamseqend"/>
      </xsl:if>
    </xsl:if>
</xsl:template>


<!--######## use_locname ########-->
<!--=============================-->
<xsl:template match="use_locname">

    <xsl:element name="hamERROR">
      <xsl:attribute name="message">
        <xsl:value-of select="'USE-LOCNAME not supported'"/>
       </xsl:attribute>
     </xsl:element>
</xsl:template>


<!--######## split_motion ########-->
<!--==============================-->
<xsl:template match="split_motion">

    <xsl:element name="hamparbegin"/>
    <xsl:apply-templates select="*[1]"/>
    <xsl:element name="hamplus"/>
    <xsl:apply-templates select="*[2]"/>
    <xsl:element name="hamparend"/>
    <xsl:call-template name="checkManner"/>
    <xsl:call-template name="checkAlternatingRepetitions"/>
    <!-- NB TBD: other general motion attributes. -->
</xsl:template>


<!--######## par_motion ########-->
<!--============================-->
<xsl:template match="par_motion">

    <xsl:element name="hamparbegin"/>
    <xsl:apply-templates select="*"/>
    <xsl:element name="hamparend"/>
    <xsl:call-template name="checkManner"/>
    <xsl:call-template name="checkAlternatingRepetitions"/>
    <!-- NB TBD: other general motion attributes. -->
</xsl:template>


<!--######## seq_motion ########-->
<!--============================-->
<xsl:template match="seq_motion">

    <!-- 2010-12:
         Bug fix for misspelling: should be "hamfusion*", rather
         than "hamseqfusion*".
    -->
    <xsl:variable name="seqfusion">
        <xsl:if test="string(@fused)!='true'">seq</xsl:if>
        <xsl:if test="string(@fused)='true'">fusion</xsl:if>
    </xsl:variable>

    <xsl:variable name="dropparens">
      <xsl:if test=
          "string(@fused)!='true' and
          (parent::*[name()='tgt_motion' or name()='split_motion' or
                    name()='seq_motion' or name()='sign_manual'])"
      >true</xsl:if>
    </xsl:variable>

    <xsl:if test="$dropparens!='true'">
      <xsl:element name="{concat('ham',$seqfusion,'begin')}"/>
    </xsl:if>
    <xsl:apply-templates select="*"/>
    <xsl:if test="$dropparens!='true'">
      <xsl:element name="{concat('ham',$seqfusion,'end')}"/>
    </xsl:if>
    <xsl:call-template name="checkManner"/>
    <xsl:call-template name="checkAlternatingRepetitions"/>
    <!-- NB TBD: other general motion attributes. -->
</xsl:template>


<!--######## tgt_motion ########-->
<!--============================-->
<xsl:template match="tgt_motion">

<!--
2011-01
Special case for <changeposture/> to deal with some (somewhat dubious)
examples, e.g. VOTER, in the Dicta-Sign LSF lexicon.
In this case we must ensure that any manner (in the tgt_motion
itself) is dealt with before any explicit target location elements.
(Do we really need to call checkBounceOnContact in this case?)
-->
    <xsl:choose>
      <xsl:when test="changeposture">
        <xsl:apply-templates
          select="*[name()='changeposture' or
                    name()='handconfig' or
                    name()='split_handconfig']"/>
        <xsl:call-template name="checkManner"/>
        <xsl:call-template name="checkBounceOnContact"/>
        <xsl:call-template name="checkAlternatingRepetitions"/>
        <xsl:apply-templates
          select="*[name()!='changeposture' and
                    name()!='handconfig' and
                    name()!='split_handconfig']"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates select="*"/>
        <xsl:call-template name="checkManner"/>
        <xsl:call-template name="checkBounceOnContact"/>
        <xsl:call-template name="checkAlternatingRepetitions"/>
      </xsl:otherwise>
    </xsl:choose>
<!--
pre 2011-01:
    <xsl:apply-templates select="*"/>
    <xsl:call-template name="checkManner"/>
    <xsl:call-template name="checkBounceOnContact"/>
-->
    <!-- NB TBD: other general motion attributes. -->
</xsl:template>


<!--######## rpt_motion ########-->
<!--============================-->
<xsl:template match="rpt_motion">

    <xsl:if test="parent::rpt_motion">
      <xsl:element name="hamparbegin"/>
    </xsl:if>

    <xsl:apply-templates select="*"/>

    <xsl:if test="not(@repetition_baseshift) and not(@ellipse_direction)">
      <!-- Case - Simple Repetition: -->
      <xsl:call-template name="doBasicRepetition">
        <xsl:with-param name="rep" select="@repetition"/>
      </xsl:call-template>
      <xsl:if test="@approx_repetition='true'">
        <xsl:element name="hametc"/>
      </xsl:if>
      <xsl:if test="@second_repetition">
        <xsl:call-template name="doBasicRepetition">
          <xsl:with-param name="rep" select="@second_repetition"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>

    <xsl:if test="(@approx_repetition='true') and (@ellipse_direction)">
      <!-- Case - Approx Repetition with constraint (2011-04): -->
      <xsl:element name="hamseqbegin"/>
      <xsl:call-template name="doBasicRepetition">
        <xsl:with-param name="rep" select="@repetition"/>
      </xsl:call-template>
      <xsl:element name="hametc"/>
      <xsl:element name="{concat('hamellipse',@ellipse_direction)}"/>
      <xsl:element name="hamseqend"/>
	</xsl:if>

	<xsl:if test="@repetition_baseshift">
      <!-- Case - Moving Repetition: -->
      <xsl:element name="hamseqbegin"/>
      <xsl:call-template name="doBasicRepetition">
        <xsl:with-param name="rep" select="@repetition"/>
      </xsl:call-template>
      <xsl:call-template name="doDirectedMove">
        <xsl:with-param name="dir" select="@repetition_baseshift"/>
      </xsl:call-template>
      <xsl:if test="@baseshift_size">
        <xsl:call-template name="doSize">
          <xsl:with-param name="sz" select="@baseshift_size"/>
        </xsl:call-template>
      </xsl:if>
      <xsl:if test="@baseshift_incrdecr">
        <xsl:element name="{concat('ham',@baseshift_incrdecr)}"/>
      </xsl:if>
      <xsl:if test="@baseshift_incrdecr_size">
        <xsl:call-template name="doSize">
          <xsl:with-param name="sz" select="@baseshift_incrdecr_size"/>
        </xsl:call-template>
      </xsl:if>
      <xsl:element name="hamseqend"/>
    </xsl:if>

    <xsl:if test="@repetition_incrdecr">
      <xsl:element name="{concat('ham',@repetition_incrdecr)}"/>
      <xsl:if test="@repetition_incrdecr_size">
        <xsl:call-template name="doSize">
          <xsl:with-param name="sz" select="@repetition_incrdecr_size"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>

    <xsl:call-template name="checkManner"/>
	<!-- 2011-01
	Now the alternating motion check is after the manner check, not
	before it.
	-->
    <xsl:call-template name="checkAlternatingRepetitions"/>
    <!-- NB TBD: other general motion attributes. -->

    <xsl:if test="parent::rpt_motion">
      <xsl:element name="hamparend"/>
    </xsl:if>
</xsl:template>


<!--######## nonman_motion ########-->
<!--===============================-->
<xsl:template match="nonman_motion">

    <xsl:element name="hamseqbegin"/>
    <xsl:apply-templates select="handconfig" mode="nonman"/>
    <xsl:apply-templates select="*[name()!='handconfig']"/>
    <xsl:element name="hamseqend"/>
    <xsl:call-template name="checkManner"/>
    <xsl:call-template name="checkAlternatingRepetitions"/>
    <!-- NB TBD: other general motion attributes. -->
</xsl:template>


<!--######## directedmotion ########-->
<!--================================-->
<xsl:template match="directedmotion">

    <xsl:call-template name="doDirectedMove">
      <xsl:with-param name="dir" select="@direction"/>
    </xsl:call-template>
    <xsl:if test="@second_direction">
      <xsl:element name="hambetween"/>
      <xsl:call-template name="doDirectedMove">
        <xsl:with-param name="dir" select="@second_direction"/>
      </xsl:call-template>
      <!--
      NB This one will never succeed for currrent HNS (and HNS-to-HML
      translation.  HNS only allows size after the second movement
      arrow (if there are two) - which the HNS-to-HML assigns to
      the size attribute (rather than to the more natural second_size).
      -->
      <xsl:call-template name="checkSecondSize"/>
    </xsl:if>
    <!-- Late call - see comment above re second_size: -->
    <xsl:call-template name="checkSize"/>
    <xsl:call-template name="checkCurve"/>
    <xsl:call-template name="checkZigZag"/>
    <xsl:if test="@ellipse_direction">
      <xsl:element name="{concat('hamellipse',@ellipse_direction)}"/>
    </xsl:if>
    <xsl:call-template name="checkManner"/>
    <xsl:call-template name="checkBounceInSpace"/>
    <!-- NB TBD: other general motion attributes. -->
    <!-- Possible brushing contact: -->
    <xsl:if test="*">
      <xsl:element name="hamseqbegin"/>
      <xsl:element name="hambrushing"/>
      <xsl:apply-templates select="*"/>
      <xsl:element name="hamseqend"/>
    </xsl:if>
    <xsl:call-template name="checkAlternatingRepetitions"/>
</xsl:template>


<!--######## circularmotion ########-->
<!--================================-->
<xsl:template match="circularmotion">

    <xsl:call-template name="doCircularMove">
      <xsl:with-param name="axis" select="@axis"/>
    </xsl:call-template>
    <xsl:if test="@second_axis">
      <xsl:element name="hambetween"/>
      <xsl:call-template name="doCircularMove">
        <xsl:with-param name="axis" select="@second_axis"/>
      </xsl:call-template>
    </xsl:if>
    <xsl:call-template name="checkSize"/>
    <!-- Revolutions: -->
    <xsl:if test="@start">
      <xsl:element name="{concat('hamclock',@start)}"/>
    </xsl:if>
    <xsl:if test="@clockplus='true'">
      <xsl:element name="hamclockfull"/>
      <xsl:if test="@second_clockplus='true'">
        <xsl:element name="hamclockfull"/></xsl:if>
    </xsl:if>
    <xsl:if test="@end">
      <xsl:element name="{concat('hamclock',@end)}"/>
    </xsl:if>
    <xsl:if test="@ellipse_direction">
      <xsl:element name="{concat('hamellipse',@ellipse_direction)}"/>
      <xsl:if test="@ellipse_size">
        <xsl:call-template name="doSize">
          <xsl:with-param name="sz" select="@ellipse_size"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>
    <xsl:call-template name="checkZigZag"/>
    <xsl:call-template name="checkIncrDecr"/>
    <xsl:call-template name="checkManner"/>
    <!-- NB TBD: other general motion attributes. -->
    <!-- Possible brushing contact: -->
    <xsl:if test="*">
      <xsl:element name="hamseqbegin"/>
      <xsl:element name="hambrushing"/>
      <xsl:apply-templates select="*"/>
      <xsl:element name="hamseqend"/>
    </xsl:if>
    <xsl:call-template name="checkAlternatingRepetitions"/>
</xsl:template>


<!--######## wristmotion ########-->
<!--=============================-->
<xsl:template match="wristmotion">

    <xsl:element name="{concat('ham',@motion)}"/>
    <xsl:call-template name="checkSize"/>
    <xsl:call-template name="checkAlternatingRepetitions"/>
</xsl:template>


<!--######## fingerplay ########-->
<!--============================-->
<xsl:template match="fingerplay">

    <xsl:choose>
      <!-- The complex case: active fingers are specified. -->
      <xsl:when test="@digits">
        <xsl:element name="hamseqbegin"/>
        <xsl:call-template name="doDigits">
          <xsl:with-param name="dd" select="@digits"/>
        </xsl:call-template>
        <xsl:element name="hamfingerplay"/>
        <xsl:element name="hamseqend"/>
      </xsl:when>
      <!-- The simple case: plain fingerplay. -->
	  <xsl:otherwise>
        <xsl:element name="hamfingerplay"/>
      </xsl:otherwise>
	</xsl:choose>
    <xsl:call-template name="checkAlternatingRepetitions"/>
</xsl:template>


<!--######## changeposture ########-->
<!--===============================-->
<xsl:template match="changeposture">

    <xsl:element name="hamreplace"/>
    <!-- 2011-04:
    Now we may have to deal with the "successivefingers" attribute
    of a following "handconfig" operand.
    -->
    <xsl:if test="following-sibling::handconfig/@successivefingers">
      <xsl:variable name="sfttag">
        <xsl:value-of
            select="substring-after(
              following-sibling::handconfig/@successivefingers,'towards_')"/>
      </xsl:variable>
      <xsl:if test="$sfttag='pinky'">
        <xsl:element name="hamincreasing"/>
      </xsl:if>
      <xsl:if test="$sfttag='thumb'">
        <xsl:element name="hamdecreasing"/>
      </xsl:if>
    </xsl:if>
</xsl:template>


<!--######## nomotion ########-->
<!--============================-->
<xsl:template match="nomotion">

    <xsl:element name="hamnomotion"/>
</xsl:template>

</xsl:transform>
